const pdfHolder = document.getElementById('pdf-data-holder');
const PDF_FILE_PATH = pdfHolder ? pdfHolder.dataset.pdfUrl : null;

console.log('📄 PDF Path:', PDF_FILE_PATH);

if (!PDF_FILE_PATH) {
    document.getElementById('loading').innerHTML = '❌ Error: PDF path not found!';
    console.error('PDF path missing from #pdf-data-holder');
}

// PDF.js Worker
pdfjsLib.GlobalWorkerOptions.workerSrc = 'https://cdnjs.cloudflare.com/ajax/libs/pdf.js/3.11.174/pdf.worker.min.js';

let pdfDoc = null;
let totalPages = 0;
let renderedPages = {};
let currentScale = 1.5;

// Start loading
window.addEventListener('load', () => {
    if (PDF_FILE_PATH) {
        console.log('🚀 Starting PDF load...');
        loadPDF(PDF_FILE_PATH);
    }
});

async function loadPDF(url) {
    document.getElementById('loading').classList.add('active');
    console.log('📥 Fetching PDF from:', url);

    try {
        const loadingTask = pdfjsLib.getDocument({
            url: url,
            cMapUrl: 'https://cdnjs.cloudflare.com/ajax/libs/pdf.js/3.11.174/cmaps/',
            cMapPacked: true
        });

        pdfDoc = await loadingTask.promise;
        totalPages = pdfDoc.numPages;
        console.log('✅ PDF loaded successfully! Total pages:', totalPages);
        await initFlipbook();
    } catch (error) {
        console.error('❌ PDF Load Error:', error);
        document.getElementById('loading').innerHTML = `
                    ❌ Failed to load PDF<br>
                    <small>${error.message}</small>
                `;
    }
}

function getResponsiveDimensions() {
    const screenWidth = window.innerWidth;
    const screenHeight = window.innerHeight;
    const isLandscape = window.matchMedia("(orientation: landscape)").matches;

    let width, height, pageWidth, pageHeight;

    // Extra small mobile (≤360px)
    if (screenWidth <= 360) {
        pageWidth = screenWidth - 20;
        pageHeight = Math.min(pageWidth * 1.4, screenHeight - 180);
        width = pageWidth;
        height = pageHeight;
    }
    // Small mobile (≤480px)
    else if (screenWidth <= 480) {
        pageWidth = screenWidth - 30;
        pageHeight = Math.min(pageWidth * 1.4, screenHeight - 180);
        width = pageWidth;
        height = pageHeight;
    }
    // Mobile portrait (≤768px)
    else if (screenWidth <= 768 && !isLandscape) {
        pageWidth = screenWidth - 40;
        pageHeight = Math.min(pageWidth * 1.35, screenHeight - 200);
        width = pageWidth;
        height = pageHeight;
    }
    // Mobile landscape (≤768px)
    else if (screenWidth <= 768 && isLandscape) {
        pageWidth = (screenWidth - 60) / 2;
        pageHeight = Math.min(screenHeight - 150, pageWidth * 1.4);
        width = pageWidth * 2;
        height = pageHeight;
    }
    // Tablet (769px - 1024px)
    else if (screenWidth <= 1024) {
        pageWidth = (screenWidth - 100) / 2;
        pageHeight = Math.min(screenHeight - 200, pageWidth * 1.4);
        width = pageWidth * 2;
        height = pageHeight;
    }
    // Desktop
    else {
        pageWidth = 500;
        pageHeight = 700;
        width = 1000;
        height = 700;
    }

    return {
        width: Math.floor(width),
        height: Math.floor(height),
        pageWidth: Math.floor(pageWidth),
        pageHeight: Math.floor(pageHeight),
        display: screenWidth <= 768 && !isLandscape ? 'single' : 'double'
    };
}

async function initFlipbook() {
    const $flipbook = $('#flipbook');
    $flipbook.empty();

    for (let i = 1; i <= totalPages; i++) {
        $flipbook.append(`<div class="page" data-page="${i}"></div>`);
    }

    const dimensions = getResponsiveDimensions();
    console.log('📐 Flipbook dimensions:', dimensions);

    // Calculate optimal scale for mobile devices
    if (window.innerWidth <= 768) {
        currentScale = window.innerWidth <= 480 ? 1.2 : 1.3;
    } else {
        currentScale = 1.5;
    }

    $flipbook.turn({
        width: dimensions.width,
        height: dimensions.height,
        autoCenter: true,
        gradients: true,
        acceleration: true,
        elevation: 50,
        duration: 1000,
        display: dimensions.display,
        when: {
            turning: (e, page) => renderPagesInView(page),
            turned: (e, page) => updatePageInfo(page)
        }
    });

    // Handle orientation and resize
    let resizeTimer;
    let lastWidth = window.innerWidth;
    let lastOrientation = window.matchMedia("(orientation: landscape)").matches;

    function handleResize() {
        clearTimeout(resizeTimer);
        resizeTimer = setTimeout(() => {
            const currentWidth = window.innerWidth;
            const currentOrientation = window.matchMedia("(orientation: landscape)").matches;

            // Only resize if width changed by more than 50px or orientation changed
            if (Math.abs(currentWidth - lastWidth) > 50 || currentOrientation !== lastOrientation) {
                const newDimensions = getResponsiveDimensions();
                console.log('🔄 Resizing to:', newDimensions);

                // Update scale for mobile
                if (window.innerWidth <= 768) {
                    currentScale = window.innerWidth <= 480 ? 1.2 : 1.3;
                } else {
                    currentScale = 1.5;
                }

                const currentPage = $flipbook.turn('page');
                $flipbook.turn('size', newDimensions.width, newDimensions.height);
                $flipbook.turn('display', newDimensions.display);

                // Re-render visible pages with new scale
                renderedPages = {};
                renderPagesInView(currentPage);

                lastWidth = currentWidth;
                lastOrientation = currentOrientation;
            }
        }, 300);
    }

    window.addEventListener('resize', handleResize);
    window.addEventListener('orientationchange', () => {
        setTimeout(handleResize, 100);
    });

    await renderPagesInView(1);
    document.getElementById('loading').classList.remove('active');
    document.getElementById('prevArrow').style.display = 'flex';
    document.getElementById('nextArrow').style.display = 'flex';
    document.getElementById('pageCounter').style.display = 'block';
    document.getElementById('totalPagesDisplay').textContent = totalPages;
    updatePageInfo(1);
}

async function renderPagesInView(currentPage) {
    const pages = [currentPage, currentPage + 1, currentPage - 1, currentPage + 2];
    for (const num of pages) {
        if (num >= 1 && num <= totalPages && !renderedPages[num]) {
            await renderPage(num);
            renderedPages[num] = true;
        }
    }
}

async function renderPage(pageNum) {
    try {
        const page = await pdfDoc.getPage(pageNum);
        const viewport = page.getViewport({
            scale: currentScale
        });

        const canvas = document.createElement('canvas');
        const context = canvas.getContext('2d');

        // Use device pixel ratio for sharper rendering on high DPI screens
        const outputScale = window.devicePixelRatio || 1;

        canvas.width = Math.floor(viewport.width * outputScale);
        canvas.height = Math.floor(viewport.height * outputScale);
        canvas.style.width = Math.floor(viewport.width) + "px";
        canvas.style.height = Math.floor(viewport.height) + "px";

        const transform = outputScale !== 1 ? [outputScale, 0, 0, outputScale, 0, 0] : null;

        await page.render({
            canvasContext: context,
            viewport: viewport,
            transform: transform
        }).promise;

        $(`#flipbook .page[data-page="${pageNum}"]`).empty().append(canvas);
        console.log(`✅ Rendered page ${pageNum}`);
    } catch (error) {
        console.error(`❌ Error rendering page ${pageNum}:`, error);
    }
}

function updatePageInfo(page) {
    const current = page || $('#flipbook').turn('page');
    document.getElementById('currentPage').textContent = current;
    document.getElementById('prevArrow').disabled = current === 1;
    document.getElementById('nextArrow').disabled = current === totalPages;
}

// Navigation
document.getElementById('prevArrow').onclick = () => $('#flipbook').turn('previous');
document.getElementById('nextArrow').onclick = () => $('#flipbook').turn('next');

// Keyboard navigation
document.addEventListener('keydown', e => {
    if (e.key === 'ArrowRight') $('#flipbook').turn('next');
    if (e.key === 'ArrowLeft') $('#flipbook').turn('previous');
});

// Touch swipe support for mobile
let touchStartX = 0;
let touchEndX = 0;

document.addEventListener('touchstart', e => {
    touchStartX = e.changedTouches[0].screenX;
}, {
    passive: true
});

document.addEventListener('touchend', e => {
    touchEndX = e.changedTouches[0].screenX;
    handleSwipe();
}, {
    passive: true
});

function handleSwipe() {
    const swipeThreshold = 50;
    if (touchEndX < touchStartX - swipeThreshold) {
        $('#flipbook').turn('next');
    }
    if (touchEndX > touchStartX + swipeThreshold) {
        $('#flipbook').turn('previous');
    }
}